﻿namespace Hims.Api.Controllers
{

    using Hims.Api.Utilities;
    using Hims.Domain.Services;
    using Hims.Shared.DataFilters;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using System.Threading.Tasks;
    using Shared.UserModels.DynamicTemplate;
    using Hims.Shared.EntityModels;

    using Hims.Shared.Library.Enums;
    using System;
    using Hims.Api.Models;

    /// <inheritdoc />
    [Authorize]
    [Route("api/dynamic-template")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class DynamicTemplateController : BaseController
    {
        /// <summary>
        /// The dynamic template services.
        /// </summary>
        private readonly IDynamicTemplateService dynamicTemplateService;
        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;
        /// <inheritdoc />
        public DynamicTemplateController(IDynamicTemplateService dynamicTemplateService, IAuditLogService auditLogServices)
        {
            this.dynamicTemplateService = dynamicTemplateService;
            this.auditLogServices = auditLogServices;
        }


        /// <summary>
        /// Inserts the dynamic template asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("insert")]
        public async Task<ActionResult> InsertAsync([FromBody] InsertModel model, [FromHeader] LocationHeader location)
        {
            model = (InsertModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.dynamicTemplateService.InsertAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DynamicTemplate,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LocationId = (int)model.LocationId,
                    LogDescription = $"{model.CreatedByName} has added <b>Dynamic Template</b> of <strong>{model.TemplateName}</strong> successfully."
                };
                await this.auditLogServices.LogAsync(auditLog);
            }
            return this.Success(response);
        }

        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch")]
        public async Task<ActionResult> FetchAllAsync([FromBody] FetchModel model)
        {
            model ??= new FetchModel();
            var response = await this.dynamicTemplateService.FetchAsync(model);
            return this.Success(response);
        }
        /// <summary>
        /// Inserts the dynamic template asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateHocAsync([FromBody] InsertModel model, [FromHeader] LocationHeader location)
        {
            model = (InsertModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.dynamicTemplateService.UpdateAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DynamicTemplate,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LocationId = (int)model.LocationId,
                    LogDescription = $"{model.CreatedByName} has updated <b>Dynamic Template</b> of <strong>{model.TemplateName}</strong> successfully."
                };
                await this.auditLogServices.LogAsync(auditLog);
            }
            return this.Success(response);
        }
        /// <summary>
        /// Deletes the asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="location">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("delete")]
        public async Task<ActionResult> DeleteAsync([FromBody] InsertModel model, [FromHeader] LocationHeader location)
        {

            model = (InsertModel)EmptyFilter.Handler(model);
            model.LocationId = !string.IsNullOrEmpty(location.LocationId) ? int.Parse(location.LocationId) : (int?)null;
            var response = await this.dynamicTemplateService.DeleteAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.DynamicTemplate,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LocationId = (int)model.LocationId,
                    LogDescription = $"{model.CreatedByName} has Deleted <b>Dynamic Template</b> of <strong>{model.TemplateName}</strong> successfully."
                };
                await this.auditLogServices.LogAsync(auditLog);
            }
            return this.Success(response);
        }

    }
}
